import numpy as np
import matplotlib.pyplot as plt
from matplotlib.backends.backend_agg import FigureCanvasAgg as FigureCanvas

def draw_sphere(ax3d, radius=1.0, offset_x=0, offset_y=0, offset_z=0,
                color='gray', alpha=0.7, angle_x=0, angle_y=0, angle_z=0):
    """Draws a sphere with given radius, position, color, alpha, and rotation."""
    u = np.linspace(0, 2 * np.pi, 30)
    v = np.linspace(0, np.pi, 30)
    u, v = np.meshgrid(u, v)
    x = radius * np.sin(v) * np.cos(u)
    y = radius * np.sin(v) * np.sin(u)
    z = radius * np.cos(v)

    # Rotation
    ax_r, ay_r, az_r = np.deg2rad([angle_x, angle_y, angle_z])
    def Rx(a): return np.array([[1, 0, 0],
                                [0, np.cos(a), -np.sin(a)],
                                [0, np.sin(a), np.cos(a)]])
    def Ry(a): return np.array([[np.cos(a), 0, np.sin(a)],
                                [0, 1, 0],
                                [-np.sin(a), 0, np.cos(a)]])
    def Rz(a): return np.array([[np.cos(a), -np.sin(a), 0],
                                [np.sin(a), np.cos(a), 0],
                                [0, 0, 1]])
    R = Rz(az_r) @ Ry(ay_r) @ Rx(ax_r)

    points = np.array([x.flatten(), y.flatten(), z.flatten()]).T @ R.T
    x, y, z = points[:, 0].reshape(x.shape) + offset_x, \
              points[:, 1].reshape(y.shape) + offset_y, \
              points[:, 2].reshape(z.shape) + offset_z

    ax3d.plot_surface(x, y, z, color=color, alpha=alpha, linewidth=0)

def draw_cylinder(ax3d, height=1.0, radius=0.1, offset_x=0, offset_y=0, offset_z=0,
                  color='gray', alpha=0.7, angle_x=0, angle_y=0, angle_z=0):
    """Draws a cylinder centered at origin along Z, then rotates and shifts it."""
    theta = np.linspace(0, 2*np.pi, 20)
    z = np.linspace(-height/2, height/2, 20)
    theta, z = np.meshgrid(theta, z)
    x = radius * np.cos(theta)
    y = radius * np.sin(theta)

    # Rotation
    ax_r, ay_r, az_r = np.deg2rad([angle_x, angle_y, angle_z])
    def Rx(a): return np.array([[1, 0, 0],
                                [0, np.cos(a), -np.sin(a)],
                                [0, np.sin(a), np.cos(a)]])
    def Ry(a): return np.array([[np.cos(a), 0, np.sin(a)],
                                [0, 1, 0],
                                [-np.sin(a), 0, np.cos(a)]])
    def Rz(a): return np.array([[np.cos(a), -np.sin(a), 0],
                                [np.sin(a), np.cos(a), 0],
                                [0, 0, 1]])
    R = Rz(az_r) @ Ry(ay_r) @ Rx(ax_r)

    points = np.array([x.flatten(), y.flatten(), z.flatten()]).T @ R.T
    x, y, z = points[:, 0].reshape(x.shape) + offset_x, \
              points[:, 1].reshape(y.shape) + offset_y, \
              points[:, 2].reshape(z.shape) + offset_z

    ax3d.plot_surface(x, y, z, color=color, alpha=alpha, linewidth=0)

def make_stickman(angle_x=0, angle_y=0, angle_z=0, img_size=400):
    """Creates a simple 3D stickman figure."""
    fig = plt.figure(figsize=(img_size / 100, img_size / 100), dpi=100)
    fig.patch.set_alpha(0.0)  # transparent
    canvas = FigureCanvas(fig)
    ax3d = fig.add_subplot(111, projection='3d')
    ax3d.set_facecolor((0, 0, 0, 0))
    ax3d.axis('off')
    ax3d.set_box_aspect([1, 1, 1])
    ax3d.view_init(elev=20, azim=30)

    color = 'gray'
    alpha = 0.8

    # Body
    draw_cylinder(ax3d, height=1.2, radius=0.25,
                  offset_x=0, offset_y=0, offset_z=0,
                  color=color, alpha=alpha)

    # Head
    draw_sphere(ax3d, radius=0.3,
                offset_x=0, offset_y=0, offset_z=0.9,
                color=color, alpha=alpha)

    # Arms
    draw_cylinder(ax3d, height=0.8, radius=0.07,
                  offset_x=0, offset_y=0.5, offset_z=0.4,
                  angle_x=90, color=color, alpha=alpha)
    draw_cylinder(ax3d, height=0.8, radius=0.07,
                  offset_x=0, offset_y=-0.5, offset_z=0.4,
                  angle_x=90, color=color, alpha=alpha)

    # Legs
    draw_cylinder(ax3d, height=1.0, radius=0.08,
                  offset_x=0, offset_y=0.2, offset_z=-0.6,
                  angle_x=0, angle_z=10, color=color, alpha=alpha)
    draw_cylinder(ax3d, height=1.0, radius=0.08,
                  offset_x=0, offset_y=-0.2, offset_z=-0.6,
                  angle_x=0, angle_z=-10, color=color, alpha=alpha)

    ax3d.set_xlim(-1.5, 1.5)
    ax3d.set_ylim(-1.5, 1.5)
    ax3d.set_zlim(-1.5, 1.5)

    canvas.draw()
    width, height = map(int, fig.get_size_inches() * fig.get_dpi())
    rgba = np.frombuffer(canvas.buffer_rgba(), dtype=np.uint8).reshape(height, width, 4)
    bgra = rgba[:, :, [2,1,0,3]].copy()
    plt.close(fig)
    return bgra

def execute(params, inputs, outputs):
    outputs.m1 = make_stickman()
    return "Hello from spiral_matplotlib! The result is in m1"

if __name__ == "__main__":
    img = make_stickman()
    from PIL import Image
    Image.fromarray(img).show()
