/*
 * Decompiled with CFR 0.152.
 */
package net.algart.matrices.tiff.tags;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.Set;
import java.util.stream.Stream;
import net.algart.matrices.tiff.TiffException;
import net.algart.matrices.tiff.TiffIFD;
import net.algart.matrices.tiff.TiffImageKind;
import net.algart.matrices.tiff.tags.TagCompression;
import net.algart.matrices.tiff.tags.TagDescription;
import net.algart.matrices.tiff.tiles.TiffMap;

public final class SvsDescription
extends TagDescription {
    public static final String SVS_IMAGE_DESCRIPTION_PREFIX = "Aperio Image";
    public static final String SVS_TYPICAL_APPLICATION = "Aperio Image Library";
    public static final String MAGNIFICATION_ATTRIBUTE = "AppMag";
    public static final String MICRON_PER_PIXEL_ATTRIBUTE = "MPP";
    public static final String LEFT_ATTRIBUTE = "Left";
    public static final String TOP_ATTRIBUTE = "Top";
    public static final String ORIGINAL_WIDTH_ATTRIBUTE = "OriginalWidth";
    public static final String ORIGINAL_HEIGHT_ATTRIBUTE = "OriginalHeight";
    public static final String SCAN_SCOPE_ID_ATTRIBUTE = "ScanScope ID";
    public static final String DATE_ATTRIBUTE = "Date";
    public static final String TIME_ATTRIBUTE = "Time";
    private static final DateTimeFormatter SVS_DATE_FORMATTER = DateTimeFormatter.ofPattern("MM/dd/yy");
    private static final DateTimeFormatter SVS_TIME_FORMATTER = DateTimeFormatter.ofPattern("HH:mm:ss");
    private static final Set<String> HUMAN_READABLE_ATTRIBUTES = Set.of("ScanScope ID", "Date", "Time");
    private final List<String> raw = new ArrayList<String>();
    private final Map<String, String> attributes = new LinkedHashMap<String, String>();
    private final String application;
    private final String summary;
    private final boolean main;

    SvsDescription(String description) {
        super(description);
        String[] lines;
        assert (description != null && description.startsWith(SVS_IMAGE_DESCRIPTION_PREFIX)) : this.getClass().getSimpleName() + " should not be constructed for " + description;
        description = description.trim();
        boolean delimiterFound = false;
        String summary = "";
        for (String line : lines = description.split("\\n")) {
            String[] records;
            String prefix;
            line = line.trim();
            this.raw.add(line);
            int p = line.indexOf(124);
            String string = prefix = p == -1 ? line : line.substring(0, p);
            if (!delimiterFound) {
                summary = prefix.trim();
            }
            if (p < 0) continue;
            delimiterFound = true;
            for (String s : records = line.substring(p + 1).split("\\|")) {
                String[] keyValue = s.trim().split("=", 2);
                if (keyValue.length != 2) continue;
                String key = keyValue[0].trim();
                String value = keyValue[1].trim();
                this.attributes.put(key, value);
            }
        }
        this.application = !this.raw.isEmpty() ? this.raw.getFirst() : "";
        this.summary = summary;
        this.main = this.hasPixelSize();
    }

    private SvsDescription(Builder builder, TiffImageKind imageKind) {
        super(Objects.requireNonNull(builder).buildDescription(imageKind));
        this.raw.add(builder.firstLine);
        this.raw.add(builder.secondLine);
        this.attributes.putAll(builder.attributes);
        this.application = builder.application;
        this.summary = builder.actualSummary;
        this.main = this.hasPixelSize();
    }

    public static boolean isSvs(String imageDescription) {
        return imageDescription != null && imageDescription.trim().startsWith(SVS_IMAGE_DESCRIPTION_PREFIX);
    }

    @Override
    public boolean isSvs() {
        return true;
    }

    public boolean isMain() {
        return this.main;
    }

    public List<String> raw() {
        return Collections.unmodifiableList(this.raw);
    }

    @Override
    public String formatName(boolean pretty) {
        return pretty ? "SVS details" : "svs";
    }

    public boolean hasApplication() {
        return !this.application.isEmpty();
    }

    @Override
    public String application() {
        return this.application;
    }

    public boolean hasSummary() {
        return !this.summary.isEmpty();
    }

    public String summary() {
        return this.summary;
    }

    @Override
    public Map<String, String> attributes() {
        return Collections.unmodifiableMap(this.attributes);
    }

    @Override
    public Set<String> humanReadableAttributeNames() {
        return HUMAN_READABLE_ATTRIBUTES;
    }

    public boolean hasPixelSize() {
        return this.attributes.containsKey(MICRON_PER_PIXEL_ATTRIBUTE);
    }

    public OptionalDouble optPixelSize() {
        return this.optDouble(MICRON_PER_PIXEL_ATTRIBUTE);
    }

    public double pixelSize() throws TiffException {
        double result = this.reqDouble(MICRON_PER_PIXEL_ATTRIBUTE);
        if (result <= 0.0) {
            throw new TiffException("SVS image description contains negative \"MPP\" attribute: " + result);
        }
        return result;
    }

    public boolean hasMagnification() {
        return this.attributes.containsKey(MAGNIFICATION_ATTRIBUTE);
    }

    public double magnification() throws TiffException {
        return this.reqDouble(MAGNIFICATION_ATTRIBUTE);
    }

    public boolean hasGeometry() {
        return this.hasPixelSize() && this.attributes.containsKey(LEFT_ATTRIBUTE) && this.attributes.containsKey(TOP_ATTRIBUTE);
    }

    public double imageLeftMicronsAxisRightward() throws TiffException {
        return this.reqDouble(LEFT_ATTRIBUTE) * 1000.0;
    }

    public double imageTopMicronsAxisUpward() throws TiffException {
        return this.reqDouble(TOP_ATTRIBUTE) * 1000.0;
    }

    @Override
    public String jsonString() {
        TiffIFD ifd;
        StringBuilder sb = new StringBuilder();
        sb.append("{\n");
        sb.append("  \"exists\": ").append(true).append(",\n");
        sb.append("  \"main\": ").append(this.main).append(",\n");
        sb.append("  \"application\": \"").append(TiffIFD.escapeJsonString(this.application())).append("\",\n");
        sb.append("  \"summary\": \"").append(TiffIFD.escapeJsonString(this.summary())).append("\",\n");
        sb.append("  \"attributes\": {\n");
        Iterator<Map.Entry<String, String>> iterator = this.attributes.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, String> entry = iterator.next();
            sb.append("    \"").append(TiffIFD.escapeJsonString(entry.getKey())).append("\": ");
            sb.append("\"").append(TiffIFD.escapeJsonString(entry.getValue())).append("\"");
            if (iterator.hasNext()) {
                sb.append(",");
            }
            sb.append("\n");
        }
        sb.append("  },\n");
        sb.append("  \"raw\": [\n");
        int n = this.raw.size();
        for (int i = 0; i < n; ++i) {
            sb.append("    \"").append(TiffIFD.escapeJsonString(this.raw.get(i))).append("\"");
            if (i < n - 1) {
                sb.append(",");
            }
            sb.append("\n");
        }
        sb.append("  ]");
        if (this.hasPixelSize()) {
            sb.append(",\n  \"pixelSize\": ");
            try {
                sb.append(this.pixelSize());
            }
            catch (TiffException e) {
                sb.append("\"format error: ").append(TiffIFD.escapeJsonString(e.getMessage())).append("\"");
            }
        }
        if ((ifd = this.getIFD()) != null && ifd.hasGlobalIndex()) {
            sb.append(",\n  \"globalIndex\": ").append(ifd.getGlobalIndex());
        }
        sb.append("\n}");
        return sb.toString();
    }

    @Override
    public String toString() {
        return this.toString(TiffIFD.StringFormat.BRIEF);
    }

    @Override
    public String toString(TiffIFD.StringFormat format) {
        Objects.requireNonNull(format, "Null format");
        if (format.isJson()) {
            return this.jsonString();
        }
        StringBuilder sb = new StringBuilder();
        if (format.isBrief()) {
            sb.append(this.main ? "SVS" : "Additional");
            if (this.hasApplication()) {
                sb.append(" [").append(this.application()).append("]");
            }
            sb.append(": ");
            sb.append(this.summary());
            OptionalDouble optional = this.optPixelSize();
            if (optional.isPresent()) {
                sb.append(", ").append(optional.getAsDouble()).append(" microns/pixel");
            }
            return sb.toString();
        }
        sb.append(this.main ? "Main image" : "Additional image");
        sb.append("%nSummary:%n  %s%n".formatted(this.summary()));
        sb.append("Attributes:");
        for (Map.Entry<String, String> entry : this.attributes.entrySet()) {
            sb.append("%n  %s = %s".formatted(entry.getKey(), entry.getValue()));
        }
        if (this.hasPixelSize()) {
            sb.append("%nPixel size: ".formatted(new Object[0]));
            try {
                sb.append(this.pixelSize());
            }
            catch (TiffException e) {
                sb.append("format error: ").append(e.getMessage());
            }
        }
        sb.append("%nRaw text:".formatted(new Object[0]));
        for (String line : this.raw) {
            sb.append("%n  %s".formatted(line));
        }
        return sb.toString();
    }

    public static Optional<SvsDescription> fromDescriptions(Stream<? extends TagDescription> descriptions) {
        Objects.requireNonNull(descriptions, "Null descriptions");
        return descriptions.filter(d -> {
            SvsDescription svs;
            return d instanceof SvsDescription && (svs = (SvsDescription)d).isMain();
        }).map(d -> (SvsDescription)d).findFirst();
    }

    public static Optional<SvsDescription> fromIFDs(Collection<? extends TiffIFD> ifds) {
        Objects.requireNonNull(ifds, "Null ifds");
        return SvsDescription.fromIFDs(ifds.stream());
    }

    public static Optional<SvsDescription> fromIFDs(Stream<? extends TiffIFD> ifds) {
        Objects.requireNonNull(ifds, "Null ifds");
        return SvsDescription.fromDescriptions(ifds.filter(Objects::nonNull).map(TiffIFD::getDescription));
    }

    public static Optional<SvsDescription> fromMaps(Collection<? extends TiffMap> maps) {
        Objects.requireNonNull(maps, "Null maps");
        return SvsDescription.fromMaps(maps.stream());
    }

    public static Optional<SvsDescription> fromMaps(Stream<? extends TiffMap> maps) {
        Objects.requireNonNull(maps, "Null maps");
        return SvsDescription.fromDescriptions(maps.filter(Objects::nonNull).map(TiffMap::description));
    }

    public static Optional<String> svsCompressionName(TagCompression compression) {
        Objects.requireNonNull(compression, "Null compression");
        if (compression.isJpeg2000()) {
            return Optional.of("J2K");
        }
        return switch (compression) {
            case TagCompression.JPEG_RGB -> Optional.of("JPEG/RGB");
            case TagCompression.JPEG -> Optional.of("JPEG/YCbCr");
            default -> Optional.empty();
        };
    }

    public static class Builder {
        private final Map<String, String> attributes = new LinkedHashMap<String, String>();
        private String application = "Aperio Image Library";
        private boolean autoGeneratedSummary = true;
        private int baseImageDimX = 0;
        private int baseImageDimY = 0;
        private int imageDimX = 0;
        private int imageDimY = 0;
        private int scanOffsetX = 0;
        private int scanOffsetY = 0;
        private int scanDimX = 0;
        private int scanDimY = 0;
        private int tileSizeX = 0;
        private int tileSizeY = 0;
        private String compressionName = null;
        private Integer quality = null;
        private String summary = null;
        private String actualSummary = null;
        private String firstLine = null;
        private String secondLine = null;

        public Builder application(String application) {
            this.application = Objects.requireNonNull(application);
            return this;
        }

        public Builder defaultApplication() {
            return this.application(SvsDescription.SVS_TYPICAL_APPLICATION);
        }

        public Builder applicationSuffix(String suffix) {
            Objects.requireNonNull(suffix);
            return this.application("Aperio Image Library " + suffix);
        }

        public Builder autoGeneratedSummary(boolean autoSummary) {
            this.autoGeneratedSummary = autoSummary;
            return this;
        }

        public Builder baseImageDimX(int baseImageDimX) {
            if (baseImageDimX <= 0) {
                throw new IllegalArgumentException("Zero or negative baseImageDimX: " + baseImageDimX);
            }
            this.baseImageDimX = baseImageDimX;
            return this;
        }

        public Builder baseImageDimY(int baseImageDimY) {
            if (baseImageDimY <= 0) {
                throw new IllegalArgumentException("Zero or negative baseImageDimY: " + baseImageDimY);
            }
            this.baseImageDimY = baseImageDimY;
            return this;
        }

        public Builder imageDimX(int imageDimX) {
            if (imageDimX <= 0) {
                throw new IllegalArgumentException("Zero or negative imageDimX: " + imageDimX);
            }
            this.imageDimX = imageDimX;
            return this;
        }

        public Builder imageDimY(int imageDimY) {
            if (imageDimY <= 0) {
                throw new IllegalArgumentException("Zero or negative imageDimY: " + imageDimY);
            }
            this.imageDimY = imageDimY;
            return this;
        }

        public Builder scanOffsetX(int scanOffsetX) {
            this.scanOffsetX = scanOffsetX;
            return this;
        }

        public Builder scanOffsetY(int scanOffsetY) {
            this.scanOffsetY = scanOffsetY;
            return this;
        }

        public Builder scanDimX(int scanDimX) {
            if (scanDimX <= 0) {
                throw new IllegalArgumentException("Zero or negative scanDimX: " + scanDimX);
            }
            this.scanDimX = scanDimX;
            return this;
        }

        public Builder scanDimY(int scanDimY) {
            if (scanDimY <= 0) {
                throw new IllegalArgumentException("Zero or negative scanDimY: " + scanDimY);
            }
            this.scanDimY = scanDimY;
            return this;
        }

        public Builder tileSizeX(int tileSizeX) {
            if (tileSizeX <= 0) {
                throw new IllegalArgumentException("Zero or negative tileSizeX: " + tileSizeX);
            }
            this.tileSizeX = tileSizeX;
            return this;
        }

        public Builder tileSizeY(int tileSizeY) {
            if (tileSizeY <= 0) {
                throw new IllegalArgumentException("Zero or negative tileSizeY: " + tileSizeY);
            }
            this.tileSizeY = tileSizeY;
            return this;
        }

        public Builder compressionName(String compressionName) {
            this.compressionName = compressionName;
            return this;
        }

        public Builder quality(Integer quality) {
            if (quality != null && quality < 0) {
                throw new IllegalArgumentException("Negative quality: " + quality);
            }
            this.quality = quality;
            return this;
        }

        public Builder summary(String summary) {
            if (this.autoGeneratedSummary) {
                throw new IllegalStateException("Cannot set summary when autoSummary is true");
            }
            this.summary = summary;
            return this;
        }

        public Builder updateFrom(TiffIFD ifd) throws TiffException {
            return this.updateFrom(ifd, true);
        }

        public Builder updateFrom(TiffIFD ifd, boolean adjustScanSizes) throws TiffException {
            Objects.requireNonNull(ifd, "Null IFD");
            this.imageDimX(ifd.getImageDimX());
            this.imageDimY(ifd.getImageDimY());
            Integer globalIndex = ifd.getGlobalIndex();
            if (globalIndex != null && globalIndex == 0) {
                this.baseImageDimX = this.imageDimX;
                this.baseImageDimY = this.imageDimY;
            }
            this.adjustMinimalScanSizes(adjustScanSizes);
            if (ifd.hasTileInformation()) {
                this.tileSizeX(ifd.getTileSizeX());
                this.tileSizeY(ifd.getTileSizeY());
            }
            ifd.optCompression().flatMap(SvsDescription::svsCompressionName).ifPresent(this::compressionName);
            return this;
        }

        public Builder attribute(String key, String value) {
            Objects.requireNonNull(key, "Null key");
            if (value == null) {
                this.attributes.remove(key);
            } else {
                this.attributes.put(key, value);
            }
            return this;
        }

        public Builder addAttributes(Map<String, String> attributes) {
            Objects.requireNonNull(attributes, "Null attributes");
            this.attributes.putAll(attributes);
            return this;
        }

        public Builder removeAttributes() {
            this.attributes.clear();
            return this;
        }

        public Builder magnification(double magnification) {
            return this.attribute(SvsDescription.MAGNIFICATION_ATTRIBUTE, Double.toString(magnification));
        }

        public Builder pixelSize(double pixelSize) {
            return this.attribute(SvsDescription.MICRON_PER_PIXEL_ATTRIBUTE, Double.toString(pixelSize));
        }

        public Builder left(double leftMicronsAxisRightward) {
            return this.attribute(SvsDescription.LEFT_ATTRIBUTE, Double.toString(leftMicronsAxisRightward));
        }

        public Builder top(double topMicronsAxisUpward) {
            return this.attribute(SvsDescription.TOP_ATTRIBUTE, Double.toString(topMicronsAxisUpward));
        }

        public Builder dateTime(LocalDateTime dateTime) {
            Objects.requireNonNull(dateTime, "Null dateTime");
            this.attribute(SvsDescription.DATE_ATTRIBUTE, SVS_DATE_FORMATTER.format(dateTime));
            this.attribute(SvsDescription.TIME_ATTRIBUTE, SVS_TIME_FORMATTER.format(dateTime));
            return this;
        }

        public SvsDescription build() {
            return this.build(TiffImageKind.ORDINARY);
        }

        public SvsDescription build(TiffImageKind imageKind) {
            Objects.requireNonNull(imageKind, "Null image imageKind");
            return new SvsDescription(this, imageKind);
        }

        private String buildDescription(TiffImageKind imageKind) {
            this.firstLine = this.application;
            String summary = this.buildSummary(imageKind);
            StringBuilder sb = new StringBuilder(summary == null ? "" : summary);
            if (imageKind == TiffImageKind.BASE || imageKind == TiffImageKind.THUMBNAIL) {
                for (Map.Entry<String, String> entry : this.attributes.entrySet()) {
                    sb.append("|");
                    sb.append(entry.getKey()).append(" = ").append(entry.getValue());
                }
            }
            this.secondLine = sb.toString();
            return this.firstLine + "\n" + this.secondLine;
        }

        private String buildSummary(TiffImageKind imageKind) {
            if (this.autoGeneratedSummary) {
                boolean baseLevel = this.imageDimX == this.baseImageDimX && this.imageDimY == this.baseImageDimY;
                switch (imageKind) {
                    case BASE: 
                    case ORDINARY: {
                        Object object = "%dx%d [%d,%d %dx%d]".formatted(this.scanDimX, this.scanDimY, this.scanOffsetX, this.scanOffsetY, this.baseImageDimX, this.baseImageDimY) + (this.tileSizeX > 0 && this.tileSizeY > 0 ? " (%dx%d)".formatted(this.tileSizeX, this.tileSizeY) : "") + (baseLevel ? "" : " -> %dx%d".formatted(this.imageDimX, this.imageDimY)) + (String)(this.compressionName != null ? " " + this.compressionName : "") + (String)(this.quality != null ? " Q=" + this.quality : "");
                        break;
                    }
                    case THUMBNAIL: {
                        Object object = "%dx%d -> %dx%d".formatted(this.baseImageDimX, this.baseImageDimY, this.imageDimX, this.imageDimY);
                        break;
                    }
                    case LABEL: 
                    case MACRO: {
                        Object object = "%s %dx%d".formatted(imageKind.keyword(), this.imageDimX, this.imageDimY);
                        break;
                    }
                    default: {
                        Object object = this.actualSummary = "";
                    }
                }
                if (this.scanDimX > 0 && this.scanDimY > 0) {
                    this.attribute(SvsDescription.ORIGINAL_WIDTH_ATTRIBUTE, String.valueOf(this.scanDimX));
                    this.attribute(SvsDescription.ORIGINAL_HEIGHT_ATTRIBUTE, String.valueOf(this.scanDimY));
                }
            } else {
                this.actualSummary = this.summary;
            }
            return this.actualSummary;
        }

        private void adjustMinimalScanSizes(boolean adjustScanSizes) {
            if (adjustScanSizes) {
                this.scanDimX = Math.max(this.scanDimX, Math.addExact(this.scanOffsetX, this.baseImageDimX));
                this.scanDimY = Math.max(this.scanDimY, Math.addExact(this.scanOffsetY, this.baseImageDimY));
            }
        }
    }
}

