/*
 * Decompiled with CFR 0.152.
 */
package net.algart.matrices.tiff.pyramids;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import net.algart.arrays.Arrays;
import net.algart.matrices.tiff.TiffException;
import net.algart.matrices.tiff.TiffIFD;
import net.algart.matrices.tiff.TiffImageKind;
import net.algart.matrices.tiff.TiffReader;
import net.algart.matrices.tiff.tags.SvsDescription;
import net.algart.matrices.tiff.tags.TagCompression;
import net.algart.matrices.tiff.tags.TagDescription;
import net.algart.matrices.tiff.tiles.TiffMap;

public final class TiffPyramidMetadata {
    public static final int SVS_THUMBNAIL_INDEX = 1;
    private static final boolean DETECT_LABEL_AND_MACRO_PAIR = Arrays.SystemSettings.getBooleanProperty((String)"net.algart.matrices.tiff.pyramids.detectLabelAndMacroPair", (boolean)true);
    private static final boolean CHECK_COMPRESSION_FOR_LABEL_AND_MACRO = Arrays.SystemSettings.getBooleanProperty((String)"net.algart.matrices.tiff.pyramids.checkCompressionForLabelAndMacro", (boolean)true);
    private static final boolean CHECK_KEYWORDS_FOR_LABEL_AND_MACRO = Arrays.SystemSettings.getBooleanProperty((String)"net.algart.matrices.tiff.pyramids.checkKeywordsForLabelAndMacro", (boolean)true);
    private static final int MAX_SPECIAL_IMAGES_SIZE = 2048;
    private static final double STANDARD_MACRO_ASPECT_RATIO = 3.0;
    private static final double ALLOWED_ASPECT_RATION_DEVIATION = 0.2;
    private static final System.Logger LOG = System.getLogger(TiffPyramidMetadata.class.getName());
    private final List<TiffIFD> ifds;
    private final int numberOfImages;
    private final int baseImageDimX;
    private final int baseImageDimY;
    private final boolean baseImageTiled;
    private final int numberOfLayers;
    private int pyramidScaleRatio = -1;
    private int thumbnailIndex = -1;
    private int labelIndex = -1;
    private int macroIndex = -1;
    private final SvsDescription svsDescription;

    private TiffPyramidMetadata() {
        this.ifds = Collections.emptyList();
        this.numberOfLayers = 0;
        this.numberOfImages = 0;
        this.baseImageDimY = 0;
        this.baseImageDimX = 0;
        this.baseImageTiled = false;
        this.svsDescription = null;
    }

    private TiffPyramidMetadata(Collection<? extends TiffIFD> ifds) throws TiffException {
        Objects.requireNonNull(ifds, "Null IFDs");
        this.ifds = List.copyOf(ifds);
        this.numberOfImages = this.ifds.size();
        this.svsDescription = SvsDescription.fromIFDs(this.ifds).orElse(null);
        if (this.numberOfImages == 0) {
            this.numberOfLayers = 0;
            this.baseImageDimY = 0;
            this.baseImageDimX = 0;
            this.baseImageTiled = false;
        } else {
            TiffIFD first = this.ifds.getFirst();
            this.baseImageDimX = first.getImageDimX();
            this.baseImageDimY = first.getImageDimY();
            this.baseImageTiled = first.hasTileInformation();
            this.numberOfLayers = this.detectPyramidAndThumbnail();
            this.detectLabelAndMacro();
        }
    }

    public static TiffPyramidMetadata empty() {
        return new TiffPyramidMetadata();
    }

    public static TiffPyramidMetadata ofMaps(Collection<? extends TiffMap> maps) throws TiffException {
        return new TiffPyramidMetadata(TiffMap.ifds(maps));
    }

    public static TiffPyramidMetadata ofIFDs(Collection<? extends TiffIFD> ifds) throws TiffException {
        return new TiffPyramidMetadata(ifds);
    }

    public static TiffPyramidMetadata of(TiffReader reader) throws IOException {
        Objects.requireNonNull(reader, "Null TIFF reader");
        return new TiffPyramidMetadata(reader.allIFDs());
    }

    public static boolean matchesDimensionRatio(long layerDimX, long layerDimY, long nextLayerDimX, long nextLayerDimY, int compression) {
        if (compression <= 1) {
            throw new IllegalArgumentException("Invalid compression " + compression + " (must be 2 or greater)");
        }
        if (layerDimX <= 0L || layerDimY <= 0L) {
            throw new IllegalArgumentException("Zero or negative layer dimensions: " + layerDimX + "x" + layerDimY);
        }
        if (nextLayerDimX <= 0L || nextLayerDimY <= 0L) {
            throw new IllegalArgumentException("Zero or negative next layer dimensions: " + nextLayerDimX + "x" + nextLayerDimY);
        }
        long predictedNextWidth = layerDimX / (long)compression;
        long predictedNextHeight = layerDimY / (long)compression;
        return !(nextLayerDimX != predictedNextWidth && nextLayerDimX != predictedNextWidth + 1L || nextLayerDimY != predictedNextHeight && nextLayerDimY != predictedNextHeight + 1L);
    }

    public static int findRatio(long layerDimX, long layerDimY, long nextLayerDimX, long nextLayerDimY) {
        for (int probeCompression = 2; probeCompression <= 32; ++probeCompression) {
            if (!TiffPyramidMetadata.matchesDimensionRatio(layerDimX, layerDimY, nextLayerDimX, nextLayerDimY, probeCompression)) continue;
            return probeCompression;
        }
        return -1;
    }

    public static void correctForSpecialKinds(TiffIFD ifd, TiffImageKind kind) {
        Objects.requireNonNull(ifd, "Null IFD");
        Objects.requireNonNull(kind, "Null image kind");
        if (kind.isOrdinary()) {
            return;
        }
        ifd.removeTileInformation();
        ifd.defaultStripSize();
        ifd.putCompression(TiffPyramidMetadata.recommendedCompression(kind));
        switch (kind) {
            case LABEL: {
                ifd.put(254, 1);
                break;
            }
            case MACRO: {
                ifd.put(254, 9);
            }
        }
    }

    public static TagCompression recommendedCompression(TiffImageKind imageKind) {
        Objects.requireNonNull(imageKind, "Null image kind");
        return imageKind == TiffImageKind.LABEL ? TagCompression.LZW : TagCompression.JPEG_RGB;
    }

    public List<TiffIFD> ifds() {
        return this.ifds;
    }

    public TiffIFD ifd(int ifdIndex) {
        return this.ifds.get(ifdIndex);
    }

    public boolean isNonTrivial() {
        return this.isSvs() || this.isPyramid();
    }

    public boolean isSvs() {
        return this.svsDescription != null;
    }

    public boolean isSvsLayer0() {
        Integer globalIndex = this.svsDescription != null ? this.svsDescription.globalIndex() : null;
        return globalIndex != null && globalIndex == 0;
    }

    public boolean isPyramid() {
        if (this.numberOfLayers == 0) {
            return false;
        }
        if (this.numberOfLayers == 1) {
            return this.isSvsLayer0();
        }
        assert (this.numberOfLayers > 1);
        return true;
    }

    public boolean isSvsCompatible() {
        return this.isSvs() || this.isPyramid() && this.hasSvsThumbnail();
    }

    public int baseImageDimX() {
        return this.baseImageDimX;
    }

    public int baseImageDimY() {
        return this.baseImageDimY;
    }

    public int numberOfImages() {
        return this.numberOfImages;
    }

    public int numberOfLayers() {
        return this.numberOfLayers;
    }

    public int pyramidScaleRatio() {
        return this.pyramidScaleRatio;
    }

    public int layerToImage(int layerIndex) {
        if (layerIndex < 0) {
            throw new IllegalArgumentException("Negative layer index " + layerIndex);
        }
        return !this.hasThumbnail() || layerIndex < this.thumbnailIndex ? layerIndex : Math.addExact(layerIndex, 1);
    }

    public int imageToLayer(int imageIndex) {
        if (imageIndex < 0) {
            throw new IllegalArgumentException("Negative IFD index " + imageIndex);
        }
        if (imageIndex >= this.numberOfImages) {
            throw new IllegalArgumentException("Too large IFD index " + imageIndex + " >= " + this.numberOfImages);
        }
        if (imageIndex == this.thumbnailIndex) {
            return -1;
        }
        int layerIndex = !this.hasThumbnail() || imageIndex < this.thumbnailIndex ? imageIndex : imageIndex - 1;
        return layerIndex < this.numberOfLayers ? layerIndex : -1;
    }

    public boolean hasSvsThumbnail() {
        return this.thumbnailIndex == 1;
    }

    public boolean hasThumbnail() {
        return this.thumbnailIndex != -1;
    }

    public int thumbnailIndex() {
        return this.thumbnailIndex;
    }

    public boolean hasLabel() {
        return this.labelIndex != -1;
    }

    public int labelIndex() {
        return this.labelIndex;
    }

    public boolean hasMacro() {
        return this.macroIndex != -1;
    }

    public int macroIndex() {
        return this.macroIndex;
    }

    public boolean isSpecial(int ifdIndex) {
        if (ifdIndex < 0) {
            throw new IllegalArgumentException("Negative IFD index = " + ifdIndex);
        }
        return ifdIndex == this.thumbnailIndex || ifdIndex == this.labelIndex || ifdIndex == this.macroIndex;
    }

    public boolean hasSpecialKinds() {
        return this.thumbnailIndex != -1 || this.labelIndex != -1 || this.macroIndex != -1;
    }

    public boolean hasSpecialKind(TiffImageKind kind) {
        return this.specialKindIndex(kind) != -1;
    }

    public int specialKindIndex(TiffImageKind kind) {
        Objects.requireNonNull(kind, "Null special kind");
        switch (kind) {
            case LABEL: {
                if (this.labelIndex == -1) break;
                return this.labelIndex;
            }
            case MACRO: {
                if (this.macroIndex == -1) break;
                return this.macroIndex;
            }
            case THUMBNAIL: {
                if (this.thumbnailIndex == -1) break;
                return this.thumbnailIndex;
            }
        }
        return -1;
    }

    public TagDescription description(int ifdIndex) {
        return this.ifd(ifdIndex).getDescription();
    }

    public SvsDescription svsDescription() {
        return this.svsDescription;
    }

    public static boolean isSmallImage(TiffIFD ifd) throws TiffException {
        return ifd.getImageDimX() <= 2048 && ifd.getImageDimY() <= 2048;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        if (this.isPyramid()) {
            sb.append("TIFF pyramid with ").append(this.numberOfLayers).append(" layers (from ").append(this.numberOfImages).append(" images) ");
            if (this.numberOfLayers > 1) {
                sb.append(this.pyramidScaleRatio).append(":1 ");
            }
            sb.append("(");
            for (int i = 0; i < this.numberOfLayers; ++i) {
                if (i > 0) {
                    sb.append("|");
                }
                sb.append(this.layerToImage(i));
            }
            sb.append(")");
        }
        for (TiffImageKind kind : TiffImageKind.values()) {
            int index;
            if (kind.isOrdinary() || (index = this.specialKindIndex(kind)) == -1) continue;
            if (!sb.isEmpty()) {
                sb.append(", ");
            }
            sb.append(kind.kindName()).append(" (#").append(index).append(")");
        }
        return sb.isEmpty() ? "no special images" : sb.toString();
    }

    private int detectPyramidAndThumbnail() throws TiffException {
        this.thumbnailIndex = -1;
        if (!this.baseImageTiled) {
            return 0;
        }
        int countNonSvs = this.detectPyramid(1);
        if (this.numberOfImages <= 1 || countNonSvs >= 3) {
            return countNonSvs;
        }
        if (TiffPyramidMetadata.isSmallImage(this.ifds.get(1))) {
            this.thumbnailIndex = 1;
        }
        if (countNonSvs == 2 && this.numberOfImages == 2) {
            return this.thumbnailIndex == -1 ? 2 : 1;
        }
        int countSvs = this.detectPyramid(2);
        if (countSvs < countNonSvs) {
            assert (countSvs == 1);
            return this.thumbnailIndex == -1 ? 2 : 1;
        }
        return countSvs;
    }

    private int detectPyramid(int startIndex) throws TiffException {
        TiffIFD ifd;
        assert (startIndex >= 1);
        int levelDimX = this.baseImageDimX;
        int levelDimY = this.baseImageDimY;
        int actualScaleRatio = -1;
        int count = 1;
        int k = startIndex;
        while (k < this.numberOfImages && (ifd = this.ifds.get(k)).isMainIFD() && ifd.hasTileInformation()) {
            int nextDimX = ifd.getImageDimX();
            int nextDimY = ifd.getImageDimY();
            if (actualScaleRatio == -1) {
                actualScaleRatio = TiffPyramidMetadata.findRatio(levelDimX, levelDimY, nextDimX, nextDimY);
                if (actualScaleRatio == -1) break;
                assert (actualScaleRatio >= 2);
            } else if (!TiffPyramidMetadata.matchesDimensionRatio(levelDimX, levelDimY, nextDimX, nextDimY, actualScaleRatio)) {
                int index = k;
                LOG.log(System.Logger.Level.DEBUG, () -> "%s found incorrect dimensions ratio at %d; skipping remaining %d IFDs".formatted(this.getClass().getSimpleName(), index, this.numberOfImages - index));
                break;
            }
            levelDimX = nextDimX;
            levelDimY = nextDimY;
            ++k;
            ++count;
        }
        this.pyramidScaleRatio = actualScaleRatio;
        int result = count;
        LOG.log(System.Logger.Level.DEBUG, () -> "%s checked dimension ratio for images 0, %d..%d%s (%d images in the series)".formatted(this.getClass().getSimpleName(), startIndex, this.numberOfImages - 1, this.pyramidScaleRatio < 0 ? " - not a pyramid" : " - found " + this.pyramidScaleRatio + ":1", result));
        return result;
    }

    private void detectLabelAndMacro() throws TiffException {
        int firstAfterPyramid = this.layerToImage(this.numberOfLayers);
        assert (firstAfterPyramid >= this.numberOfLayers);
        if (!this.isPyramid()) {
            return;
        }
        assert (this.numberOfLayers > 0) : "isPyramid()=true without layers";
        if (this.numberOfImages <= 2) {
            return;
        }
        if (!this.detectTwoLastImages(firstAfterPyramid)) {
            this.detectSingleLastImage(firstAfterPyramid);
        }
    }

    private boolean detectTwoLastImages(int firstAfterPyramid) throws TiffException {
        boolean found;
        assert (firstAfterPyramid >= 1);
        if (!DETECT_LABEL_AND_MACRO_PAIR) {
            return false;
        }
        if (this.numberOfImages - firstAfterPyramid < 2) {
            return false;
        }
        assert (this.numberOfImages >= 3);
        int index1 = this.numberOfImages - 2;
        int index2 = this.numberOfImages - 1;
        TiffIFD ifd1 = this.ifds.get(index1);
        TiffIFD ifd2 = this.ifds.get(index2);
        if (!TiffPyramidMetadata.isPossibleLabelOrMacro(ifd1) || !TiffPyramidMetadata.isPossibleLabelOrMacro(ifd2)) {
            return false;
        }
        if (LOG.isLoggable(System.Logger.Level.DEBUG)) {
            LOG.log(System.Logger.Level.DEBUG, "Checking last 2 small IFDs #%d %s and #%d %s for LABEL and MACRO...".formatted(index1, TiffPyramidMetadata.sizesToString(ifd1), index2, TiffPyramidMetadata.sizesToString(ifd2)));
        }
        if (this.isSvs() && CHECK_COMPRESSION_FOR_LABEL_AND_MACRO) {
            found = false;
            int compression1 = ifd1.getCompressionCode();
            int compression2 = ifd2.getCompressionCode();
            if (compression1 == 5 && compression2 == 7) {
                this.labelIndex = index1;
                this.macroIndex = index2;
                found = true;
            }
            if (compression1 == 7 && compression2 == 5) {
                this.labelIndex = index2;
                this.macroIndex = index1;
                found = true;
            }
            if (found) {
                LOG.log(System.Logger.Level.DEBUG, () -> "LABEL %d / MACRO %d detected by compression according to the Aperio specification".formatted(this.labelIndex, this.macroIndex));
                return true;
            }
        }
        if (CHECK_KEYWORDS_FOR_LABEL_AND_MACRO) {
            found = false;
            boolean probablyLabel1 = TiffPyramidMetadata.containsLabelMarker(ifd1);
            boolean probablyMacro1 = TiffPyramidMetadata.containsMacroMarker(ifd1);
            boolean probablyLabel2 = TiffPyramidMetadata.containsLabelMarker(ifd2);
            boolean probablyMacro2 = TiffPyramidMetadata.containsMacroMarker(ifd2);
            if (probablyLabel1 && !probablyMacro1 && probablyMacro2 && !probablyLabel2) {
                this.labelIndex = index1;
                this.macroIndex = index2;
                found = true;
            }
            if (probablyMacro1 && !probablyLabel1 && probablyLabel2 && !probablyMacro2) {
                this.labelIndex = index2;
                this.macroIndex = index1;
                found = true;
            }
            if (found) {
                LOG.log(System.Logger.Level.DEBUG, () -> "LABEL %d / MACRO %d detected by keywords 'label'/'macro'".formatted(this.labelIndex, this.macroIndex));
                return true;
            }
        }
        if (ifd1.isReducedImage() && ifd2.isReducedImage()) {
            double ratio1 = TiffPyramidMetadata.ratio(ifd1);
            double ratio2 = TiffPyramidMetadata.ratio(ifd2);
            double maxRatio = Math.max(ratio1, ratio2);
            LOG.log(System.Logger.Level.DEBUG, () -> "Last 2 IFD ratios: %.5f for %d, %.5f for %d, standard MACRO %.5f".formatted(ratio1, index1, ratio2, index2, 3.0));
            if (maxRatio > 2.4000000000000004 && maxRatio < 3.75) {
                this.macroIndex = ratio1 > ratio2 ? index1 : index2;
                this.labelIndex = ratio1 > ratio2 ? index2 : index1;
                LOG.log(System.Logger.Level.DEBUG, () -> "LABEL %d / MACRO %d detected by aspect ratio".formatted(this.labelIndex, this.macroIndex));
                return true;
            }
        }
        return false;
    }

    private void detectSingleLastImage(int firstAfterPyramid) throws TiffException {
        if (this.numberOfImages - firstAfterPyramid < 1) {
            return;
        }
        assert (this.numberOfImages >= 3) : "numberOfImages<=2 was checked before calling this method";
        int index = this.numberOfImages - 1;
        TiffIFD ifd = this.ifds.get(index);
        if (!TiffPyramidMetadata.isPossibleLabelOrMacro(ifd)) {
            return;
        }
        double ratio = TiffPyramidMetadata.ratio(ifd);
        if (LOG.isLoggable(System.Logger.Level.DEBUG)) {
            LOG.log(System.Logger.Level.DEBUG, "Checking last 1 small IFDs #%d %s for LABEL or MACRO...".formatted(index, TiffPyramidMetadata.sizesToString(ifd)));
        }
        if (CHECK_KEYWORDS_FOR_LABEL_AND_MACRO) {
            boolean probablyLabel = TiffPyramidMetadata.containsLabelMarker(ifd);
            boolean probablyMacro = TiffPyramidMetadata.containsMacroMarker(ifd);
            if (probablyLabel && !probablyMacro) {
                this.labelIndex = index;
                LOG.log(System.Logger.Level.DEBUG, () -> "LABEL %d detected by keyword 'label'".formatted(this.labelIndex));
                return;
            }
            if (probablyMacro && !probablyLabel) {
                this.macroIndex = index;
                LOG.log(System.Logger.Level.DEBUG, () -> "MACRO %d detected by keyword 'macro'".formatted(this.macroIndex));
                return;
            }
        }
        if (ifd.isReducedImage()) {
            LOG.log(System.Logger.Level.DEBUG, () -> String.format("Last IFD #%d, ratio: %.5f, standard Macro %.5f", index, ratio, 3.0));
            if (ratio > 2.4000000000000004 && ratio < 3.75) {
                this.macroIndex = index;
                LOG.log(System.Logger.Level.DEBUG, () -> "MACRO %d detected by aspect ratio".formatted(this.macroIndex));
            }
        }
    }

    private static boolean isPossibleLabelOrMacro(TiffIFD ifd) throws TiffException {
        return TiffPyramidMetadata.isSmallImage(ifd) && !ifd.hasTileInformation();
    }

    private static boolean containsLabelMarker(TiffIFD ifd) throws TiffException {
        Optional<String> description = ifd.optDescription();
        return description.isPresent() && TiffImageKind.LABEL_DETECTION_PATTERN.matcher(description.get()).find();
    }

    private static boolean containsMacroMarker(TiffIFD ifd) throws TiffException {
        Optional<String> description = ifd.optDescription();
        return description.isPresent() && TiffImageKind.MACRO_DETECTION_PATTERN.matcher(description.get()).find();
    }

    private static String sizesToString(TiffIFD ifd) throws TiffException {
        Objects.requireNonNull(ifd, "Null IFD");
        return ifd.getImageDimX() + "x" + ifd.getImageDimY();
    }

    private static double ratio(TiffIFD ifd) throws TiffException {
        long dimX = ifd.getImageDimX();
        long dimY = ifd.getImageDimY();
        assert (dimX > 0L && dimY > 0L);
        return (double)Math.max(dimX, dimY) / (double)Math.min(dimX, dimY);
    }

    public static void main(String[] args) throws IOException {
        System.out.println(TiffPyramidMetadata.containsLabelMarker(new TiffIFD().putDescription("My Label is")));
        System.out.println(TiffPyramidMetadata.containsLabelMarker(new TiffIFD().putDescription("Label")));
        System.out.println(TiffPyramidMetadata.containsLabelMarker(new TiffIFD().putDescription("Labelling")));
        System.out.println(TiffPyramidMetadata.containsLabelMarker(new TiffIFD().putDescription("Label\n1")));
        System.out.println(TiffPyramidMetadata.containsLabelMarker(new TiffIFD().putDescription("Labe\n1")));
    }
}

